// Copyright (c) 2020 Metrolab Technology S.A., Geneva, Switzerland (www.metrolab.com)
// See the included file LICENSE.txt for the licensing conditions.

//////////////////////////////////////////////////////////////////////////
/// \file
/// \brief Test THM1176 API: Status manipulation.

#include "gtest/gtest.h"
#include "THM1176TestUtilities.h"
#include <ctime>

using namespace MTL::Instrument;
using namespace MTL::Instrument::THM1176Types;

/// \brief Test THM1176 API: Status manipulation.
class CTHM1176StatusTest : public ::testing::Test
{
protected:
	static THM1176_TEST_RESOURCE_MANAGER_CLASS * pResourceManager;
	static CTHM1176Instrument<THM1176_TEST_INSTRUMENT_CLASS, THM1176_TEST_RESOURCE_MANAGER_CLASS> * pTHM1176;

	static void SetUpTestCase()
	{
		ASSERT_EQ(true, ConnectToTHM1176(pResourceManager, pTHM1176));
		ASSERT_NE(nullptr, pResourceManager);
		ASSERT_NE(nullptr, pTHM1176);
		ASSERT_EQ(true, pTHM1176->IsOpen());
		ASSERT_EQ(true, pTHM1176->Reset());
	}
	
	static void TearDownTestCase()
	{
		delete pTHM1176;
		pTHM1176 = nullptr;
		delete pResourceManager;
		pResourceManager = nullptr;
	}
};
THM1176_TEST_RESOURCE_MANAGER_CLASS * CTHM1176StatusTest::pResourceManager = nullptr;
CTHM1176Instrument<THM1176_TEST_INSTRUMENT_CLASS, THM1176_TEST_RESOURCE_MANAGER_CLASS> * CTHM1176StatusTest::pTHM1176 = nullptr;

/// \test Test StatusSetEnableRegister method for all four SCPI register sets.
TEST_F(CTHM1176StatusTest, StatusSetEnableRegister)
{
	CSCPIBuffer l_ReadBuffer;
	
	{
		uStatusByte l_DisableMask, l_EnableMask;
		l_DisableMask.RawSTB = 0xff;
		l_EnableMask.StatusByte.EAV = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("4\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));

		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSTB = 0x00;
		l_EnableMask.StatusByte.QSB = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("8\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));

		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSTB = 0x00;
		l_EnableMask.StatusByte.MAV = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("16\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));

		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSTB = 0x00;
		l_EnableMask.StatusByte.ESB = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("32\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));

		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSTB = 0x00;
		l_EnableMask.StatusByte.RQS = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("0\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end())); // Special case; see IEEE 488.2 10.34.3 and 10.35.3.

		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSTB = 0x00;
		l_EnableMask.StatusByte.OSB = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("128\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));

		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSTB = 0x00;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte, l_DisableMask.RawSTB, l_EnableMask.RawSTB));
		ASSERT_EQ(true, pTHM1176->Write("*SRE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("0\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
	}
	{
		uStandardEvent l_DisableMask, l_EnableMask;
		l_DisableMask.RawSE = 0xff;
		l_EnableMask.StandardEvent.OperationComplete = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("1\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSE = 0x00;
		l_EnableMask.StandardEvent.QueryError = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("4\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSE = 0x00;
		l_EnableMask.StandardEvent.DeviceDependentError = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("8\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSE = 0x00;
		l_EnableMask.StandardEvent.ExecutionError = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("16\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSE = 0x00;
		l_EnableMask.StandardEvent.CommandError = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("32\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSE = 0x00;
		l_EnableMask.StandardEvent.PowerOn = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("128\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawSE = 0x00;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister, l_DisableMask.RawSE, l_EnableMask.RawSE));
		ASSERT_EQ(true, pTHM1176->Write("*ESE?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("0\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
	}
	{
		uOPERation l_DisableMask, l_EnableMask;
		l_DisableMask.RawOPER = 0xffff;
		l_EnableMask.OPERation.CALibrating = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister, l_DisableMask.RawOPER, l_EnableMask.RawOPER));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:QUES:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("1\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawOPER = 0x0000;
		l_EnableMask.OPERation.RANGing = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister, l_DisableMask.RawOPER, l_EnableMask.RawOPER));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:QUES:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("4\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawOPER = 0x0000;
		l_EnableMask.OPERation.MEASuring = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister, l_DisableMask.RawOPER, l_EnableMask.RawOPER));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:QUES:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("16\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawOPER = 0x0000;
		l_EnableMask.OPERation.WaitingForTRIGger = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister, l_DisableMask.RawOPER, l_EnableMask.RawOPER));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:QUES:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("32\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawOPER = 0x0000;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister, l_DisableMask.RawOPER, l_EnableMask.RawOPER));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:QUES:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("0\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
	}
	{
		uQUEStionable l_DisableMask, l_EnableMask;
		l_DisableMask.RawQUES = 0xffff;
		l_EnableMask.QUEStionable.Frequency = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister, l_DisableMask.RawQUES, l_EnableMask.RawQUES));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:OPER:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("32\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawQUES = 0x0000;
		l_EnableMask.QUEStionable.Overrange = 1;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister, l_DisableMask.RawQUES, l_EnableMask.RawQUES));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:OPER:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("512\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
		
		l_DisableMask = l_EnableMask;
		l_EnableMask.RawQUES = 0x0000;
		ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister, l_DisableMask.RawQUES, l_EnableMask.RawQUES));
		ASSERT_EQ(true, pTHM1176->Write(":STAT:OPER:ENAB?") && pTHM1176->Read(l_ReadBuffer));
		EXPECT_EQ("0\n", std::string(l_ReadBuffer.begin(), l_ReadBuffer.end()));
	}
}

/// \test Test StatusGet method (single register variant) on all three register types of all four SCPI register sets.
TEST_F(CTHM1176StatusTest, StatusGetSingleRegister)
{
	uStatusByte		l_StatusByte;
	uStandardEvent	l_StandardEvent;
	uOPERation		l_Operation;
	uQUEStionable	l_Questionable;

	// Test getting enable registers.
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte,						0xff, 0x01));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister,		0xff, 0x02));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister,	0xff, 0x04));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister,0xff, 0x08));
	
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusByte, kStatusEnable), l_StatusByte.RawSTB));
	EXPECT_EQ(0x01, l_StatusByte.RawSTB);
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStandardEventStatusRegister, kStatusEnable), l_StandardEvent.RawSE));
	EXPECT_EQ(0x02, l_StandardEvent.RawSE);
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusOperationStatusRegister, kStatusEnable), l_Operation.RawOPER));
	EXPECT_EQ(0x04, l_Operation.RawOPER);
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusQuestionableStatusRegister, kStatusEnable), l_Questionable.RawQUES));
	EXPECT_EQ(0x08, l_Questionable.RawQUES);
	
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte,						0xff, 0x00));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister,		0xff, 0x00));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister,	0xff, 0x00));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister,0xff, 0x00));
	
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusByte, kStatusEnable), l_StatusByte.RawSTB));
	EXPECT_EQ(0x00, l_StatusByte.RawSTB);
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStandardEventStatusRegister, kStatusEnable), l_StandardEvent.RawSE));
	EXPECT_EQ(0x00, l_StandardEvent.RawSE);
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusOperationStatusRegister, kStatusEnable), l_Operation.RawOPER));
	EXPECT_EQ(0x00, l_Operation.RawOPER);
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusQuestionableStatusRegister, kStatusEnable), l_Questionable.RawQUES));
	EXPECT_EQ(0x00, l_Questionable.RawQUES);
	
	// Perform a measurement, with an overrun error, to set some status bits.
	std::string l_Command = "*RST;*CLS;*ESE 8;:STAT:OPER:ENAB 16;:STAT:QUES:ENAB 32;:SENS:AUTO 0;UPP MAX;:TRIG:SOUR TIM;TIM MIN;COUN 2;:AVER:COUN 10;:INIT;*OPC";
	ASSERT_EQ(true, pTHM1176->Write(l_Command));
	
	// Test getting condition registers.
	ASSERT_EQ(false, pTHM1176->StatusGet(sStatusRegister(kStatusByte, kStatusCondition), l_StatusByte.RawSTB));
	
	ASSERT_EQ(false, pTHM1176->StatusGet(sStatusRegister(kStandardEventStatusRegister, kStatusCondition), l_StandardEvent.RawSE));
	
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusOperationStatusRegister, kStatusCondition), l_Operation.RawOPER));
	EXPECT_EQ(0, l_Operation.RawOPER);

	std::vector<sError> l_ErrorList = pTHM1176->CurrentErrorList();
	EXPECT_EQ(1, l_ErrorList.size());
	EXPECT_EQ(206, l_ErrorList[0].Code);
	EXPECT_EQ("Timer was overrun", l_ErrorList[0].Description);
	EXPECT_EQ(":STAT:OPER:COND?", l_ErrorList[0].Context);
	
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusQuestionableStatusRegister, kStatusCondition), l_Questionable.RawQUES));
	EXPECT_EQ(0, l_Questionable.RawQUES);

	// Test getting event registers.
	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusByte, kStatusEvent), l_StatusByte.RawSTB));
	EXPECT_EQ(0, l_StatusByte.StatusByte.EAV);
	EXPECT_EQ(1, l_StatusByte.StatusByte.QSB);
	EXPECT_EQ(0, l_StatusByte.StatusByte.MAV);
	EXPECT_EQ(1, l_StatusByte.StatusByte.ESB);
	EXPECT_EQ(0, l_StatusByte.StatusByte.RQS);
	EXPECT_EQ(1, l_StatusByte.StatusByte.OSB);

	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStandardEventStatusRegister, kStatusEvent), l_StandardEvent.RawSE));
	EXPECT_EQ(1, l_StandardEvent.StandardEvent.OperationComplete);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.QueryError);
	EXPECT_EQ(1, l_StandardEvent.StandardEvent.DeviceDependentError);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.ExecutionError);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.CommandError);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.PowerOn);

	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusOperationStatusRegister, kStatusEvent), l_Operation.RawOPER));
	EXPECT_EQ(0, l_Operation.OPERation.CALibrating);
	EXPECT_EQ(1, l_Operation.OPERation.RANGing);
	EXPECT_EQ(1, l_Operation.OPERation.MEASuring);
	EXPECT_EQ(1, l_Operation.OPERation.WaitingForTRIGger);

	ASSERT_EQ(true, pTHM1176->StatusGet(sStatusRegister(kStatusQuestionableStatusRegister, kStatusEvent), l_Questionable.RawQUES));
	EXPECT_EQ(1, l_Questionable.QUEStionable.Frequency);
	EXPECT_EQ(0, l_Questionable.QUEStionable.Overrange);
}

/// \test Test StatusGet method (multiple register variant) on all three register types of all four SCPI register sets.
TEST_F(CTHM1176StatusTest, StatusGetMultipleRegisters)
{
	RegisterList		l_Registers;
	StatusValues		l_Status;
	
	// Test getting enable registers.
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte,						0xff, 0x01));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister,		0xff, 0x02));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister,	0xff, 0x04));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister,0xff, 0x08));
	
	l_Registers.clear();
	l_Registers.push_back(sStatusRegister(kStatusByte,						kStatusEnable));
	l_Registers.push_back(sStatusRegister(kStandardEventStatusRegister,		kStatusEnable));
	l_Registers.push_back(sStatusRegister(kStatusOperationStatusRegister,	kStatusEnable));
	l_Registers.push_back(sStatusRegister(kStatusQuestionableStatusRegister,kStatusEnable));
	l_Status.clear();
	
	ASSERT_EQ(true, pTHM1176->StatusGet(l_Registers, l_Status));
	EXPECT_EQ(4, l_Status.size());
	EXPECT_EQ(0x01, l_Status[0]);
	EXPECT_EQ(0x02, l_Status[1]);
	EXPECT_EQ(0x04, l_Status[2]);
	EXPECT_EQ(0x08, l_Status[3]);
	
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusByte,						0xff, 0x00));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStandardEventStatusRegister,		0xff, 0x00));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister,	0xff, 0x00));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister,0xff, 0x00));
	
	ASSERT_EQ(true, pTHM1176->StatusGet(l_Registers, l_Status));
	EXPECT_EQ(4, l_Status.size());
	EXPECT_EQ(0x00, l_Status[0]);
	EXPECT_EQ(0x00, l_Status[1]);
	EXPECT_EQ(0x00, l_Status[2]);
	EXPECT_EQ(0x00, l_Status[3]);

	// Perform a measurement, with an overrun error, to set some status bits.
	std::string l_Command = "*RST;*CLS;*ESE 8;:STAT:OPER:ENAB 16;:STAT:QUES:ENAB 32;:SENS:AUTO 0;UPP MAX;:TRIG:SOUR TIM;TIM MIN;COUN 2;:AVER:COUN 10;:INIT;*OPC";
	ASSERT_EQ(true, pTHM1176->Write(l_Command));
	
	// Test getting condition registers.
	l_Registers.clear();
	l_Registers.push_back(sStatusRegister(kStatusByte,						kStatusCondition));
	l_Registers.push_back(sStatusRegister(kStandardEventStatusRegister,		kStatusCondition));
	l_Registers.push_back(sStatusRegister(kStatusOperationStatusRegister,	kStatusCondition));
	l_Registers.push_back(sStatusRegister(kStatusQuestionableStatusRegister,kStatusCondition));
	l_Status.clear();
	
	ASSERT_EQ(false, pTHM1176->StatusGet(l_Registers, l_Status));
	
	l_Registers.clear();
	l_Registers.push_back(sStatusRegister(kStatusOperationStatusRegister,	kStatusCondition));
	l_Registers.push_back(sStatusRegister(kStatusQuestionableStatusRegister,kStatusCondition));
	l_Status.clear();
	
	ASSERT_EQ(true, pTHM1176->StatusGet(l_Registers, l_Status));
	EXPECT_EQ(2, l_Status.size());
	EXPECT_EQ(0x00, l_Status[0]);
	EXPECT_EQ(0x00, l_Status[1]);
	
	std::vector<sError> l_ErrorList = pTHM1176->CurrentErrorList();
	EXPECT_EQ(1, l_ErrorList.size());
	EXPECT_EQ(206, l_ErrorList[0].Code);
	EXPECT_EQ("Timer was overrun", l_ErrorList[0].Description);
	EXPECT_EQ(":STAT:OPER:COND?;:STAT:QUES:COND?", l_ErrorList[0].Context);
	
	// Test getting event registers.
	l_Registers.clear();
	l_Registers.push_back(sStatusRegister(kStatusByte,						kStatusEvent));
	l_Registers.push_back(sStatusRegister(kStandardEventStatusRegister,		kStatusEvent));
	l_Registers.push_back(sStatusRegister(kStatusOperationStatusRegister,	kStatusEvent));
	l_Registers.push_back(sStatusRegister(kStatusQuestionableStatusRegister,kStatusEvent));
	l_Status.clear();
	
	ASSERT_EQ(true, pTHM1176->StatusGet(l_Registers, l_Status));
	EXPECT_EQ(4, l_Status.size());

	uStatusByte		l_StatusByte;
	l_StatusByte.RawSTB = l_Status[0];
	EXPECT_EQ(0, l_StatusByte.StatusByte.EAV);
	EXPECT_EQ(1, l_StatusByte.StatusByte.QSB);
	EXPECT_EQ(0, l_StatusByte.StatusByte.MAV);
	EXPECT_EQ(1, l_StatusByte.StatusByte.ESB);
	EXPECT_EQ(0, l_StatusByte.StatusByte.RQS);
	EXPECT_EQ(1, l_StatusByte.StatusByte.OSB);
	
	uStandardEvent	l_StandardEvent;
	l_StandardEvent.RawSE = l_Status[1];
	EXPECT_EQ(1, l_StandardEvent.StandardEvent.OperationComplete);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.QueryError);
	EXPECT_EQ(1, l_StandardEvent.StandardEvent.DeviceDependentError);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.ExecutionError);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.CommandError);
	EXPECT_EQ(0, l_StandardEvent.StandardEvent.PowerOn);
	
	uOPERation		l_Operation;
	l_Operation.RawOPER = l_Status[2];
	EXPECT_EQ(0, l_Operation.OPERation.CALibrating);
	EXPECT_EQ(1, l_Operation.OPERation.RANGing);
	EXPECT_EQ(1, l_Operation.OPERation.MEASuring);
	EXPECT_EQ(1, l_Operation.OPERation.WaitingForTRIGger);
	
	uQUEStionable	l_Questionable;
	l_Questionable.RawQUES = l_Status[3];
	EXPECT_EQ(1, l_Questionable.QUEStionable.Frequency);
	EXPECT_EQ(0, l_Questionable.QUEStionable.Overrange);
}

/// \test Test StatusPreset method.
TEST_F(CTHM1176StatusTest, StatusPreset)
{
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusOperationStatusRegister,	0xff, 0x04));
	ASSERT_EQ(true, pTHM1176->StatusSetEnableRegister(kStatusQuestionableStatusRegister,0xff, 0x08));
	
	ASSERT_EQ(true, pTHM1176->StatusPreset());
	
	RegisterList		l_Registers;
	StatusValues		l_Status;
	l_Registers.clear();
	l_Registers.push_back(sStatusRegister(kStatusOperationStatusRegister,	kStatusEnable));
	l_Registers.push_back(sStatusRegister(kStatusQuestionableStatusRegister,kStatusEnable));
	l_Status.clear();
	ASSERT_EQ(true, pTHM1176->StatusGet(l_Registers, l_Status));
	EXPECT_EQ(2, l_Status.size());
	EXPECT_EQ(0x00, l_Status[0]);
	EXPECT_EQ(0x00, l_Status[1]);

}
