// Copyright (c) 2020 Metrolab Technology S.A., Geneva, Switzerland (www.metrolab.com)
// See the included file LICENSE.txt for the licensing conditions.

//////////////////////////////////////////////////////////////////////////
/// \file
/// \brief Test THM1176 API: File access

#include "gtest/gtest.h"
#include "THM1176TestUtilities.h"
#include <ctime>

using namespace MTL::Instrument;
using namespace MTL::Instrument::THM1176Types;

/// \brief Test THM1176 API: File access
class CTHM1176FileAccessTest : public ::testing::Test
{
protected:
	static THM1176_TEST_RESOURCE_MANAGER_CLASS * pResourceManager;
	static CTHM1176Instrument<THM1176_TEST_INSTRUMENT_CLASS, THM1176_TEST_RESOURCE_MANAGER_CLASS> * pTHM1176;

	static void SetUpTestCase()
	{
		ASSERT_EQ(true, ConnectToTHM1176(pResourceManager, pTHM1176));
		ASSERT_NE(nullptr, pResourceManager);
		ASSERT_NE(nullptr, pTHM1176);
		ASSERT_EQ(true, pTHM1176->IsOpen());
		ASSERT_EQ(true, pTHM1176->Reset());
	}
	
	static void TearDownTestCase()
	{
		delete pTHM1176;
		pTHM1176 = nullptr;
		delete pResourceManager;
		pResourceManager = nullptr;
	}
};
THM1176_TEST_RESOURCE_MANAGER_CLASS * CTHM1176FileAccessTest::pResourceManager = nullptr;
CTHM1176Instrument<THM1176_TEST_INSTRUMENT_CLASS, THM1176_TEST_RESOURCE_MANAGER_CLASS> * CTHM1176FileAccessTest::pTHM1176 = nullptr;

/// \test Test ReadFileDirectory method.
TEST_F(CTHM1176FileAccessTest, ReadFileDirectory)
{
	U32			l_UsedBytes;
	U32			l_AvailableBytes;
	tFileList	l_FileList;
	
	ASSERT_EQ(true, pTHM1176->ReadFileDirectory(l_UsedBytes, l_AvailableBytes, l_FileList));
	EXPECT_LT(0U, l_UsedBytes);
	EXPECT_EQ(0, l_AvailableBytes);
	EXPECT_EQ(4, l_FileList.size());
	
	EXPECT_EQ("cal.dat", l_FileList[0].Path);
	EXPECT_EQ("help.dat", l_FileList[1].Path);
	EXPECT_EQ("info.dat", l_FileList[2].Path);
	EXPECT_EQ("zero.dat", l_FileList[3].Path);
	
	EXPECT_EQ("BINARY", l_FileList[0].Type);
	EXPECT_EQ("ASCII", l_FileList[1].Type);
	EXPECT_EQ("BINARY", l_FileList[2].Type);
	EXPECT_EQ("BINARY", l_FileList[3].Type);
	
	for (auto l_File : l_FileList)
		EXPECT_LT(0, l_FileList[3].Size);
}

/// \test Test ReadFile method.
TEST_F(CTHM1176FileAccessTest, ReadFile)
{
	U32			l_UsedBytes;
	U32			l_AvailableBytes;
	tFileList	l_FileList;
	std::string l_Content;
	
	ASSERT_EQ(true, pTHM1176->ReadFileDirectory(l_UsedBytes, l_AvailableBytes, l_FileList));
	
	ASSERT_EQ(true, pTHM1176->ReadFile(l_FileList[0].Path, l_Content));
	EXPECT_EQ("Calibration",	l_Content.substr(0, 11));
	ASSERT_EQ(true, pTHM1176->ReadFile(l_FileList[1].Path, l_Content));
	EXPECT_EQ("Help",			l_Content.substr(0, 4));
	ASSERT_EQ(true, pTHM1176->ReadFile(l_FileList[2].Path, l_Content));
	EXPECT_EQ("Info",			l_Content.substr(0, 4));
	ASSERT_EQ(true, pTHM1176->ReadFile(l_FileList[3].Path, l_Content));
	EXPECT_EQ("Zero",			l_Content.substr(0, 4));
}

/// \test Test ReadInformationDates method.
TEST_F(CTHM1176FileAccessTest, ReadInformationDates)
{
	std::string l_SManufacturingDate;
	std::time_t l_ManufacturingDate;
	std::string l_SCalibrationDate;
	std::time_t l_CalibrationDate;
	std::time_t	l_CurrentTime = std::time(NULL);
	static const F64 THM1176_MAX_MFG = 60. * 60. * 24. * 365.25 * 20.;	// 20 years in seconds
	static const F64 THM1176_MAX_CAL = 60. * 60. * 24. * 365.25 * 3.;	// 3 years in seconds
	
	ASSERT_EQ(true, pTHM1176->ReadInformationDates(l_SManufacturingDate, l_ManufacturingDate, l_SCalibrationDate, l_CalibrationDate));
	EXPECT_LT(0.,				difftime(l_CurrentTime, l_ManufacturingDate));
	EXPECT_GT(THM1176_MAX_MFG,	difftime(l_CurrentTime, l_ManufacturingDate));
	EXPECT_LT(0.,				difftime(l_CurrentTime, l_CalibrationDate));
	EXPECT_GT(THM1176_MAX_CAL,	difftime(l_CurrentTime, l_CalibrationDate));
}
