// Copyright (c) 2020 Metrolab Technology S.A., Geneva, Switzerland (www.metrolab.com)
// See the included file LICENSE.txt for the licensing conditions.

//////////////////////////////////////////////////////////////////////////
/// \file
/// \brief Implementation for VISAInstrument QtTest object.

#include "TestVISAInstrument.h"

#define PT2026_USB_RESOURCE_PATTERN		"USB[0-9]*::0x1BFA::0x07EA::[0-9]+::INSTR"
#define USB_RESOURCE_PATTERN			"USB[0-9]*::?*"

// Local Utilities
void l_PromptUser(std::string S)
{
	std::cout << "PROMPT : " << S << std::endl;
	std::cin.ignore();
}

// Mandatory methods
void TestVISAInstrument::initTestCase()
{
	// Initialize resource manager
	QVERIFY(m_RM.Initialize());

	// Find a PT2026 resource
	l_PromptUser("Plug one instrument of type " PT2026_USB_RESOURCE_PATTERN);
	CResourceList l_RSL;
	QVERIFY(m_RM.FindResources(l_RSL, PT2026_USB_RESOURCE_PATTERN));
	QCOMPARE(l_RSL.size(), (size_t)1);
}
void TestVISAInstrument::cleanupTestCase()
{
	l_PromptUser("Press Enter to quit");
}

// Test cases
/// \test Test FindResources.
void TestVISAInstrument::FindResources()
{
	l_PromptUser("Plug at least one instrument");
	CResourceList l_RSL;
	QVERIFY(m_RM.FindResources(l_RSL) == true);
	QVERIFY(!l_RSL.empty());
}

/// \test FindResourcesWithFilter.
void TestVISAInstrument::FindResourcesWithFilter()
{
	l_PromptUser("Plug at least one instrument of type " PT2026_USB_RESOURCE_PATTERN);
	CResourceList l_RSL;
	QVERIFY(m_RM.FindResources(l_RSL, "USB[0-9]*::0x1BFA::0x07EA::[0-9]+::INSTR") == true);
	QVERIFY(!l_RSL.empty());
}

/// \test Test ResourceInfo.
void TestVISAInstrument::ResourceInfo()
{
	l_PromptUser("Plug at least one instrument");
	CResourceList l_RSL;
	QVERIFY(m_RM.FindResources(l_RSL) == true);
	QVERIFY(!l_RSL.empty());

	eInterfaceType l_Type;
	tInterfaceNumber l_Num;
	QVERIFY(m_RM.ResourceInfo(l_RSL.front(), l_Type, l_Num));

	VISAResourceInfo l_RsrcInfo;
	QVERIFY(m_RM.ResourceInfo(l_RSL.front(), l_RsrcInfo));

	QCOMPARE(l_Type, l_RsrcInfo.Type());
	QCOMPARE(l_Num, l_RsrcInfo.Number());
}

/// \test Test basic communication with instrument.
void TestVISAInstrument::QueryIDN()
{
	l_PromptUser("Plug one USB instrument with *IDN? capability");
	CResourceList l_RSL;
	QVERIFY(m_RM.FindResources(l_RSL, USB_RESOURCE_PATTERN) == true);
	QCOMPARE(l_RSL.size(), (size_t)1);

	CVISAInstrument l_Instr(m_RM, l_RSL.front());
	// Communicate
	QVERIFY(l_Instr.Open() == true);
	QVERIFY(l_Instr.Write("*IDN?\n") == true);
	CVISABuffer l_Answer;
	QVERIFY(l_Instr.Read(l_Answer) == true);
	QVERIFY(l_Answer.size() > 0);
}

// Test main
QTEST_MAIN(TestVISAInstrument)
