//////////////////////////////////////////////////////////////////////////
/// \file
/// \brief Collection of utility macros for error messages.

#pragma once

//---------------------------------------------------------------------------//
//	Standard includes
//---------------------------------------------------------------------------//
#include <iostream>			// ostream
#include <sstream>			// std::cerr, std::cout, std::wcout
#include <cassert>			// assert
#include <thread>			// std::thread
#include <chrono>			// std::chrono

//---------------------------------------------------------------------------//
//	String helpers
//---------------------------------------------------------------------------//
#define MTL__STR_HELPER__(x)		#x
#define MTL__STR__(x)				MTL__STR_HELPER__(x)
#define MTL__LOCATION_HELPER__		__FILE__ ":" MTL__STR__(__LINE__)
#define MTL__LOCATION__				(MTL__LOCATION_HELPER__)

//---------------------------------------------------------------------------//
//	Output helpers
//---------------------------------------------------------------------------//
#ifndef MTL_MULTITHREAD_APPLICATION
#define CERR(__X__)			std::cerr << __X__
#define COUT(__X__)			std::cout << __X__
#define WCOUT(__X__)		std::wcout << __X__
#else
// "Synchronized" output helpers : allow to define thread safe methods, redirection to files, etc... To be defined in user code
extern inline void MTLLockOutputStream();
extern inline void MTLUnlockOutputStream();
#define CERR(__X__)		{MTLLockOutputStream(); std::cerr << __X__; MTLUnlockOutputStream();}
#define COUT(__X__)		{MTLLockOutputStream(); std::cout << __X__; MTLUnlockOutputStream();}
#define WCOUT(__X__)	{MTLLockOutputStream(); std::wcout << __X__; MTLUnlockOutputStream();}
#endif

//---------------------------------------------------------------------------//
//	Other helpers
//---------------------------------------------------------------------------//
// Definitions
#define MTL_Assert					assert
#define MTL_StaticAssert			static_assert
#define MTL_SleepMs(_ms_)			std::this_thread::sleep_for(std::chrono::milliseconds(_ms_))
#define MTL_Unused(x)				(void)x;
